#include "handle_monitor_cmd/handle_monitor_cmd.h"

#include "utils/logger.h"
#include "automounter_api.h"
#include "automounter_api_info.h"
#include "automounter_api_events.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <signal.h>
#include <poll.h>
#include <errno.h>

static error_code_t handle_monitor_cmd_parse_args(int argc, char *argv[]);
static error_code_t handle_monitor_cmd_parse_loglevel(int argc,char *argv[],int *arg_cntr_ptr);
static void handle_monitor_cmd_print_help(void);
static void handle_monitor_cmd_print_usage(void);

static error_code_t handle_monitor_cmd_start(void);
static void handle_monitor_cmd_sigterm_handler(int a_signal);
static error_code_t handle_monitor_cmd_mainloop(void);

static void handle_monitor_cmd_on_establish_connection_success(void);
static void handle_monitor_cmd_on_connection_lost(void);
static void handle_monitor_cmd_on_establish_connection_failure(void);

static void handle_monitor_cmd_on_part_state_changed(const partition_info_t *part_info,
		const device_info_t *device_info);
static void handle_monitor_cmd_on_dev_state_changed(const device_info_t *dev_info);

static void handle_monitor_cmd_on_partinfo_update(const partition_info_t *part_info,
		const device_info_t *device_info, int request_id);
static void handle_monitor_cmd_on_devinfo_update(const device_info_t *dev_info, int request_id);
static void handle_monitor_cmd_on_snapshot_done(int request_id);

static error_code_t handle_monitor_cmd_try_connect(void);

static void handle_monitor_cmd_print_part_info(const partition_info_t *part_info,const device_info_t *device_info);
static void handle_monitor_cmd_print_dev_info(const device_info_t *device_info);

automounterctl_command_vtable_t handle_monitor_cmd_vtable=
{
		.command_description="Prints all events received from the automounter daemon.",
		.command="monitor",
		.init=NULL,
		.parse_args=handle_monitor_cmd_parse_args,
		.start=handle_monitor_cmd_start,
		.deinit=NULL
};

static automounter_api_callbacks_t handle_monitor_cmd_event_callbacks =
{
		.on_establish_connection_success=handle_monitor_cmd_on_establish_connection_success,
		.on_connection_lost=handle_monitor_cmd_on_connection_lost,
		.on_establish_connection_failure=handle_monitor_cmd_on_establish_connection_failure,
		.on_device_detected=handle_monitor_cmd_on_dev_state_changed,
		.on_device_nomedia=handle_monitor_cmd_on_dev_state_changed,
		.on_device_automounting=handle_monitor_cmd_on_dev_state_changed,
		.on_device_automounted=handle_monitor_cmd_on_dev_state_changed,
		.on_device_unmounting=handle_monitor_cmd_on_dev_state_changed,
		.on_device_unmounted=handle_monitor_cmd_on_dev_state_changed,
		.on_device_invalid=handle_monitor_cmd_on_dev_state_changed,
		.on_partition_detected=handle_monitor_cmd_on_part_state_changed,
		.on_partition_unsupported=handle_monitor_cmd_on_part_state_changed,
		.on_partition_mounting=handle_monitor_cmd_on_part_state_changed,
		.on_partition_mounted=handle_monitor_cmd_on_part_state_changed,
		.on_partition_mount_err=handle_monitor_cmd_on_part_state_changed,
		.on_partition_remounting=handle_monitor_cmd_on_part_state_changed,
		.on_partition_unmounting=handle_monitor_cmd_on_part_state_changed,
		.on_partition_unmounted=handle_monitor_cmd_on_part_state_changed,
		.on_partition_invalid=handle_monitor_cmd_on_part_state_changed,
		.on_update_device_info=handle_monitor_cmd_on_devinfo_update,
		.on_update_partition_info=handle_monitor_cmd_on_partinfo_update,
		.on_snapshot_complete=handle_monitor_cmd_on_snapshot_done
};

static logger_loglevel_t loglevel=LOGGER_LEVEL_ERROR;
static bool verbose_output=false;
static bool application_running=true;

static error_code_t handle_monitor_cmd_parse_args(int argc, char *argv[])
{
	int arg_cntr=2;
	error_code_t result=RESULT_OK;

	//no additional parameter given, so we can go on with the command
	if (argc==2)
		return RESULT_OK;

	while (arg_cntr<argc && result==RESULT_OK)
	{
		if (strcmp(argv[arg_cntr],"--help")==0 || strcmp(argv[arg_cntr],"-h")==0)
		{
			handle_monitor_cmd_print_help();
			result=RESULT_HELP_PRINTED;
		}
		else if (strcmp(argv[arg_cntr],"--loglevel")==0 || strcmp(argv[arg_cntr],"-l")==0)
		{
			result=handle_monitor_cmd_parse_loglevel(argc,argv,&arg_cntr);
		}
		else if (strcmp(argv[arg_cntr],"--verbose")==0 || strcmp(argv[arg_cntr],"-v")==0)
		{
			verbose_output=true;
		}
		else
		{
			printf("ERROR: Invalid command line option %s.\n", argv[arg_cntr]);
			handle_monitor_cmd_print_usage();
			result=RESULT_INVALID_ARGS;
		}
		arg_cntr++;
	}

	return result;
}

static error_code_t handle_monitor_cmd_parse_loglevel(int argc,char *argv[],int *arg_cntr_ptr)
{
	*arg_cntr_ptr=*arg_cntr_ptr+1;

	if (*arg_cntr_ptr>=argc)
	{
		printf("ERROR: Log level expected after --loglevel.\n");
		handle_monitor_cmd_print_usage();
		return RESULT_INVALID_ARGS;
	}

	if (logger_parse_loglevel(argv[*arg_cntr_ptr], &loglevel)!=RESULT_OK)
	{
		printf("ERROR: Unknown log level %s.\n",argv[*arg_cntr_ptr]);
		handle_monitor_cmd_print_usage();
		return RESULT_INVALID_ARGS;
	}

	return RESULT_OK;
}

static void handle_monitor_cmd_print_help(void)
{
	printf("\n");
	printf("Automounter Control Utility - Used to control the ADIT automounter daemon.\n\n");
	printf("automounterctl %s [OPTIONS]\n",handle_monitor_cmd_vtable.command);
	printf("\t-h,--help:\t\tdisplays this help and exits.\n");
	printf("\t-l,--loglevel error|info|debug:\t\tSets the log level of the automounter shared library.\n");
	printf("\t-v,--verbose Prints all attributes of devices and partitions with the events received.\n\n");
	printf("This command prints all events received from the automounter daemon.\n\n");
}

static void handle_monitor_cmd_print_usage(void)
{
	printf("\n");
	printf("Usage: automounterctl %s --help | [--loglevel <level>] [--verbose]\n",
			handle_monitor_cmd_vtable.command);
	printf("\t-h,--help:\t\t\t\t displays this help and exits.\n");
	printf("\t-l,--loglevel error | info | debug:\t\t Sets the log level of the automounter shared library.\n");
	printf("\t-v,--verbose \t\t\t\t Prints all attributes of devices and partitions with the events received.\n");
	printf("\n");
}

static error_code_t handle_monitor_cmd_start(void)
{
	error_code_t result;

	result=automounter_api_init("automounterctl",loglevel,true);
	automounter_api_register_callbacks(&handle_monitor_cmd_event_callbacks);

	if (result==RESULT_OK)
		result=handle_monitor_cmd_try_connect();

	if (result==RESULT_OK)
	{
		signal(SIGTERM, handle_monitor_cmd_sigterm_handler);
		signal(SIGINT, handle_monitor_cmd_sigterm_handler);
		result=handle_monitor_cmd_mainloop();
		automounter_api_disconnect();
	}

	automounter_api_deinit();
	return result;
}

static void handle_monitor_cmd_sigterm_handler(int a_signal)
{
	(void)a_signal;
	//if we are hanging somewhere, simple kick us out on the second try
	if (!application_running)
		exit(1);
	application_running=false;
}

static error_code_t handle_monitor_cmd_mainloop(void)
{
	error_code_t result=RESULT_OK;
	int am_pollfd;
	struct pollfd am_pollfd_struc;

	am_pollfd=automounter_api_get_pollfd();
	if (am_pollfd==-1)
		return RESULT_INVALID;

	am_pollfd_struc.fd=am_pollfd;
	am_pollfd_struc.events=POLLIN;

	while (result==RESULT_OK && application_running)
	{
		if (poll(&am_pollfd_struc,1,-1)<=0)
		{
			if (errno!=EINTR)
			{
				printf("ERROR: Automounterctl detected problems with its poll file descriptor in the main loop.\n");
				result=RESULT_NORESOURCE;
			}
			else
				application_running=false;
			continue;
		}

		if (verbose_output)
			printf("AUTOMOUNTERCTL - Mainloop got woken up to dispatch events.\n");
		automounter_api_dispatch_event();
	}
	return result;
}

static error_code_t handle_monitor_cmd_try_connect(void)
{
	error_code_t result;

	result=automounter_api_try_connect();
	if (result==RESULT_OK)
	{
		if (automounter_api_get_state()==AM_API_INITIALIZED_WAITING_FOR_DAEMON)
			printf("Waiting for the daemon to appear.\n");
		else
			printf("Connection to automounter daemon established.\n");
	}
	else
	{
		printf("Connection to automounter daemon failed.\n");
	}

	return result;
}

static void handle_monitor_cmd_on_establish_connection_success(void)
{
	int request_id;
	printf("EVENT: Connection to automounter established.\n");
	printf("I'm going to request a snapshot.\n");
	if (automounter_api_get_snapshot(SNAPSHOT_COMPLETE,&request_id)!=RESULT_OK)
		printf("The request failed.\n");
	printf("Request with id=%d sent successfully.\n",request_id);
}

static void handle_monitor_cmd_on_connection_lost()
{
	printf("EVENT: Connection to automounter lost.\n");
	printf("Trying to reconnect.\n");
	if (handle_monitor_cmd_try_connect()!=RESULT_OK)
	{
		printf("Unable to reconnect with the automounter.\n");
		application_running=false;
	}
}

static void handle_monitor_cmd_on_establish_connection_failure(void)
{
	printf("EVENT: Failed connecting to automounter.\n");
	application_running=false;
}

static void handle_monitor_cmd_on_part_state_changed(const partition_info_t *part_info,
		const device_info_t *device_info)
{
	printf("EVENT: State of partition %s changed to %s.\n", part_info->interface_id,
			automounter_api_get_partition_state_string(part_info->state));
	if (verbose_output)
		handle_monitor_cmd_print_part_info(part_info,device_info);
}

static void handle_monitor_cmd_on_partinfo_update(const partition_info_t *part_info,
		const device_info_t *device_info, int request_id)
{
	printf("EVENT: Partition %s update received (request_id=%d).\n", part_info->interface_id,request_id);
	if (verbose_output)
		handle_monitor_cmd_print_part_info(part_info,device_info);
}

static void handle_monitor_cmd_print_part_info(const partition_info_t *part_info,const device_info_t *device_info)
{
	printf("---------------------------------------------------------------------------------\n");
	printf("\tInterface ID: %s\n",part_info->interface_id);
	printf("\tState: %s\n",automounter_api_get_partition_state_string(part_info->state));
	printf("\tIdentifier: %s\n",part_info->identifier);
	printf("\tMountpoint: %s\n",part_info->mount_point);
	printf("\tMount source: %s\n",part_info->mount_src);
	printf("\tFilesystem: %s\n",part_info->mount_fs);
	printf("\tMounted writable: %d\n",part_info->mounted_writable);
	printf("\tPartition number: %d\n",part_info->partition_no);
	if (part_info->state==PARTITION_UNSUPPORTED)
		printf("\tUnsupported reason: %s\n",
				automounter_api_get_partition_unsupported_reason_string(part_info->unsupported_reason));
	if (part_info->error != RESULT_OK)
		printf("\tError Code: %d\n",part_info->error);
	printf("\tParent device: %s\n",device_info->interface_id);
	printf("\t\tState: %s\n",automounter_api_get_device_state_string(device_info->state));
	printf("\t\tIdentifier: %s\n",device_info->identifier);
	printf("\t\tDetected partitions: %d\n",device_info->detected_partition_cnt);
	printf("\t\tDevice handler ID: %s\n",device_info->device_handler_id);
	printf("\t\tDevice type: %s\n",device_info->device_type);
	printf("---------------------------------------------------------------------------------\n");
}

static void handle_monitor_cmd_on_dev_state_changed(const device_info_t *dev_info)
{
	printf("EVENT: State of device %s changed to %s.\n", dev_info->interface_id,
			automounter_api_get_device_state_string(dev_info->state));
	if (verbose_output)
		handle_monitor_cmd_print_dev_info(dev_info);
}

static void handle_monitor_cmd_on_devinfo_update(const device_info_t *dev_info, int request_id)
{
	printf("EVENT: Device %s update received (request_id=%d).\n", dev_info->interface_id,request_id);
	if (verbose_output)
		handle_monitor_cmd_print_dev_info(dev_info);
}

static void handle_monitor_cmd_print_dev_info(const device_info_t *device_info)
{
	printf("---------------------------------------------------------------------------------\n");
	printf("\tInterface ID: %s\n",device_info->interface_id);
	printf("\tState: %s\n",automounter_api_get_device_state_string(device_info->state));
	printf("\tIdentifier: %s\n",device_info->identifier);
	printf("\tDetected partitions: %d\n",device_info->detected_partition_cnt);
	printf("\tDevice handler ID: %s\n",device_info->device_handler_id);
	printf("\tDevice type: %s\n",device_info->device_type);
	printf("---------------------------------------------------------------------------------\n");
}

static void handle_monitor_cmd_on_snapshot_done(int request_id)
{
	printf("Snapshot request (request_id=%d) done.\n",request_id);
}
